import { sql } from "drizzle-orm";
import { pgTable, text, varchar } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

export const users = pgTable("users", {
  id: varchar("id").primaryKey().default(sql`gen_random_uuid()`),
  username: text("username").notNull().unique(),
  password: text("password").notNull(),
});

export const insertUserSchema = createInsertSchema(users).pick({
  username: true,
  password: true,
});

export type InsertUser = z.infer<typeof insertUserSchema>;
export type User = typeof users.$inferSelect;

export const creatorRewardsSchema = z.object({
  walletAddress: z.string().min(32).max(44),
  claimedRewards: z.string(),
  claimedRewardsUsd: z.string(),
  unclaimedRewards: z.string(),
  unclaimedRewardsUsd: z.string(),
  feesForBuyback: z.string(),
  feesForBuybackUsd: z.string(),
  solPrice: z.number(),
});

export type CreatorRewards = z.infer<typeof creatorRewardsSchema>;

export const rewardsQuerySchema = z.object({
  walletAddress: z.string().min(32, "Invalid wallet address").max(44, "Invalid wallet address"),
});

export type RewardsQuery = z.infer<typeof rewardsQuerySchema>;

export const marketCapSchema = z.object({
  tokenAddress: z.string(),
  marketCapUsd: z.string(),
  marketCapFormatted: z.string(),
  tokenPriceUsd: z.string(),
  fdv: z.string(),
});

export type MarketCap = z.infer<typeof marketCapSchema>;
