import { useEffect, useState, useRef } from "react";
import { useToast } from "@/hooks/use-toast";
import { useRewards, useMarketCap } from "@/hooks/use-buyback";
import logoUrl from "@assets/logo.png";
import { Loader2 } from "lucide-react";

const DEFAULT_WALLET = "CGDoQB28JSB6A4zugdfaQE9K94feSopUEVm6xwVuk3Zs";
const CONTRACT_ADDRESS = "4aQwPYGCntVKpFCEUwVB1qpxmxhBsLF56mSKL5njpump";
const PUMP_PROFILE_URL = `https://pump.fun/profile/${DEFAULT_WALLET}`;
const PUMP_TOKEN_URL = `https://pump.fun/coin/${CONTRACT_ADDRESS}`;
const TELEGRAM_URL = "https://t.me/buybacksol";
const TWITTER_URL = "https://x.com/i/communities/2001598101371412726";

export default function Home() {
  const { toast } = useToast();
  const [navOpen, setNavOpen] = useState(false);
  const navRef = useRef<HTMLElement>(null);

  const { data: rewards, refetch, isFetching } = useRewards(DEFAULT_WALLET);
  const { data: marketCapData } = useMarketCap(CONTRACT_ADDRESS);

  useEffect(() => {
    // Initial fetch handled by hook, but explicit refetch if needed by UI
    refetch();
  }, [refetch]);

  useEffect(() => {
    const handleEscape = (e: KeyboardEvent) => {
      if (e.key === "Escape") setNavOpen(false);
    };
    document.addEventListener("keydown", handleEscape);
    return () => document.removeEventListener("keydown", handleEscape);
  }, []);

  const copyToClipboard = async (text: string, label: string) => {
    try {
      await navigator.clipboard.writeText(text);
      toast({ title: "Copied", description: `${label} copied to clipboard` });
    } catch {
      toast({ title: "Failed to copy", variant: "destructive" });
    }
  };

  const scrollToSection = (id: string) => {
    setNavOpen(false);
    document.getElementById(id)?.scrollIntoView({ behavior: "smooth" });
  };

  return (
    <div className="min-h-screen dark bg-[#0b0b10] text-white/90 font-sans">
      <style>{`
        .hero-art {
          position: relative;
          min-height: 460px;
          border-radius: 18px;
          border: 1px solid rgba(255, 255, 255, 0.10);
          background: linear-gradient(180deg, rgba(255, 255, 255, 0.06) 0%, rgba(255, 255, 255, 0.02) 100%);
          overflow: hidden;
          box-shadow: 0 24px 60px rgba(0, 0, 0, 0.45);
          transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .hero-art:hover {
          transform: scale(1.02);
          box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
        }
        .hero-art::before {
          content: "";
          position: absolute;
          top: 0;
          left: -100%;
          width: 100%;
          height: 2px;
          background: linear-gradient(90deg, transparent, #a855f7);
          animation: animate1 3s linear infinite;
        }
        .hero-art::after {
          content: "";
          position: absolute;
          top: -100%;
          right: 0;
          width: 2px;
          height: 100%;
          background: linear-gradient(180deg, transparent, #ec4899);
          animation: animate2 3s linear infinite;
          animation-delay: 0.75s;
        }
        .neon-border-1 {
          position: absolute;
          bottom: 0;
          right: -100%;
          width: 100%;
          height: 2px;
          background: linear-gradient(270deg, transparent, #7c3aed);
          animation: animate3 3s linear infinite;
          animation-delay: 1.5s;
        }
        .neon-border-2 {
          position: absolute;
          bottom: -100%;
          left: 0;
          width: 2px;
          height: 100%;
          background: linear-gradient(360deg, transparent, #a855f7);
          animation: animate4 3s linear infinite;
          animation-delay: 2.25s;
        }
        .orb {
          position: absolute;
          border-radius: 999px;
          filter: blur(20px);
          opacity: 0.85;
        }
        .orb-1 {
          width: 240px;
          height: 240px;
          left: -60px;
          top: 40px;
          background: rgba(168, 85, 247, 0.55);
        }
        .orb-2 {
          width: 280px;
          height: 280px;
          right: -80px;
          bottom: -80px;
          background: rgba(236, 72, 153, 0.42);
        }
        .gridlines {
          position: absolute;
          inset: 0;
          background-image: linear-gradient(to right, rgba(255, 255, 255, 0.06) 1px, transparent 1px),
            linear-gradient(to bottom, rgba(255, 255, 255, 0.06) 1px, transparent 1px);
          background-size: 46px 46px;
          opacity: 0.18;
          mask-image: radial-gradient(circle at 40% 30%, rgba(0, 0, 0, 0.95), transparent 65%);
        }
        .snake-neon-box {
          position: relative;
          overflow: hidden;
        }
        .snake-neon-box::before {
          content: "";
          position: absolute;
          top: 0;
          left: -100%;
          width: 100%;
          height: 2px;
          background: linear-gradient(90deg, transparent, #a855f7);
          animation: animate1 3s linear infinite;
        }
        .snake-neon-box::after {
          content: "";
          position: absolute;
          top: -100%;
          right: 0;
          width: 2px;
          height: 100%;
          background: linear-gradient(180deg, transparent, #ec4899);
          animation: animate2 3s linear infinite;
          animation-delay: 0.75s;
        }
      `}</style>

      <header className="sticky top-0 z-50 bg-[rgba(11,11,16,0.78)] backdrop-blur-md border-b border-white/8">
        <div className="max-w-[1120px] mx-auto px-5 py-3.5 flex items-center justify-between gap-4">
          <a href="#home" className="flex items-center gap-2.5" onClick={(e) => { e.preventDefault(); scrollToSection("home"); }}>
            <img src={logoUrl} alt="Buyback logo" width="40" height="40" className="rounded-xl shadow-lg transition-transform duration-300 hover:scale-110" />
            <span className="font-bold tracking-tight text-lg">Buyback</span>
          </a>

          <button
            className="md:hidden h-11 w-12 rounded-xl border border-white/12 bg-white/6 flex items-center justify-center hover:bg-white/10 transition-colors"
            onClick={() => setNavOpen(!navOpen)}
            aria-label="Toggle menu"
            data-testid="button-nav-toggle"
          >
            <div className="space-y-1.5">
              <span className="block h-0.5 w-4.5 bg-white/85 rounded-full" style={{ width: 18 }}></span>
              <span className="block h-0.5 w-4.5 bg-white/85 rounded-full" style={{ width: 18 }}></span>
            </div>
          </button>

          <nav ref={navRef} className={`${navOpen ? "flex" : "hidden"} md:flex flex-col md:flex-row absolute md:relative top-[68px] md:top-auto left-5 right-5 md:left-auto md:right-auto gap-1.5 md:gap-3 items-stretch md:items-center p-3 md:p-0 rounded-2xl md:rounded-none bg-[rgba(11,11,16,0.92)] md:bg-transparent border border-white/10 md:border-0 shadow-lg md:shadow-none backdrop-blur-lg md:backdrop-blur-none`}>
            <NavLink onClick={() => scrollToSection("home")}>Home</NavLink>
            <NavLink onClick={() => scrollToSection("copy-ca")}>Copy CA</NavLink>
            <NavLink onClick={() => scrollToSection("creator-reward")}>Creator Reward</NavLink>
            <NavLink onClick={() => scrollToSection("tokenomics")}>Tokenomics</NavLink>
            <NavLink onClick={() => scrollToSection("community")}>Join Community</NavLink>
            <button onClick={() => scrollToSection("copy-ca")} className="btn-gradient text-sm font-bold px-3.5 py-2.5 rounded-xl border border-white/10 transition-transform hover:scale-105 active:scale-95" data-testid="button-nav-ca">
              Get CA
            </button>
          </nav>
        </div>
      </header>

      <main className="gradient-bg">
        <section id="home" className="py-16 md:py-20">
          <div className="max-w-[1120px] mx-auto px-5 grid md:grid-cols-2 gap-9 items-center">
            <div>
              <p className="inline-flex items-center gap-2.5 px-3 py-2 rounded-full bg-white/6 border border-white/10 text-white/85 font-semibold text-sm mb-3.5 hover:bg-white/10 transition-colors cursor-default">
                on Solana
              </p>
              <h1 className="text-4xl md:text-6xl lg:text-7xl font-bold leading-none tracking-tight mb-3.5 font-display" data-testid="text-hero-title">
                Buyback.
                <span className="accent-gradient block">Hold. Build. Repeat.</span>
              </h1>
              <p className="text-white/70 text-base leading-relaxed max-w-lg mb-5">
                All fees are used to buyback. Every 1k will dropped on the chart to help support and sent it where it belongs. 1000x
              </p>

              <div className="flex flex-wrap gap-3 mb-5">
                <a href={TELEGRAM_URL} target="_blank" rel="noreferrer" className="btn-gradient font-bold px-4 py-3 rounded-xl border border-white/10 transition-transform hover:scale-105 active:scale-95 inline-block" data-testid="link-community">
                  Join Community
                </a>
                <a href={PUMP_TOKEN_URL} target="_blank" rel="noreferrer" className="relative overflow-visible font-bold px-4 py-3 rounded-xl bg-gradient-to-r from-purple-500/20 to-pink-500/20 border border-white/20 transition-all hover:scale-105 hover:bg-white/10 active:scale-95 inline-block text-white" data-testid="link-pump">
                  Buy on Pump.fun
                </a>
              </div>

              <div className="grid grid-cols-3 gap-3">
                <div className="panel rounded-xl p-4 hover:border-white/20 transition-colors">
                  <div className="font-bold font-display">Fast</div>
                  <div className="text-white/55 text-sm mt-1">Solana-native</div>
                </div>
                <div className="panel rounded-xl p-4 hover:border-white/20 transition-colors">
                  <div className="font-bold font-display">Community</div>
                  <div className="text-white/55 text-sm mt-1">Driven growth</div>
                </div>
                <div className="panel rounded-xl p-4 hover:border-white/20 transition-colors">
                  <div className="font-bold font-display">Buyback</div>
                  <div className="text-white/55 text-sm mt-1">Fees for Buyback</div>
                </div>
              </div>
            </div>

            <div className="hero-art hidden md:block" aria-hidden="true">
              <div className="neon-border-1"></div>
              <div className="neon-border-2"></div>
              <div className="orb orb-1"></div>
              <div className="orb orb-2"></div>
              <div className="absolute inset-0 m-auto w-[min(320px,76%)] h-40 grid grid-cols-[112px_1fr] gap-4 items-center p-5 rounded-2xl bg-[rgba(15,16,32,0.65)] border border-white/12 shadow-2xl backdrop-blur-lg hover:border-white/20 transition-colors">
                <img src={logoUrl} alt="" width="112" height="112" className="rounded-xl object-cover transition-transform duration-300 hover:scale-110" />
                <div>
                  <div className="text-2xl font-bold tracking-tight font-display">Buyback</div>
                  <div className="text-white/70 mt-1.5">on Solana</div>
                </div>
              </div>
              <div className="gridlines"></div>
            </div>
          </div>
        </section>

        <section id="copy-ca" className="py-20">
          <div className="max-w-[1120px] mx-auto px-5">
            <div className="mb-6">
              <h2 className="text-2xl md:text-4xl font-bold tracking-tight font-display" data-testid="text-ca-title">Copy Contract Address</h2>
            </div>

            <div className="panel rounded-2xl p-5 hover:border-white/20 transition-colors relative overflow-hidden snake-neon-box">
              <div className="neon-border-1"></div>
              <div className="neon-border-2"></div>
              <div className="flex items-center justify-between gap-3 mb-4">
                <span className="font-extrabold tracking-tight text-base font-display">CA</span>
                <span className="text-white/55 text-sm">Click copy to save it to clipboard</span>
              </div>
              <div className="flex gap-3 items-center">
                <code className="panel-dark rounded-xl px-4 py-3 text-white/90 overflow-hidden text-ellipsis whitespace-nowrap flex-1 block font-mono text-sm md:text-base" data-testid="text-ca-value">
                  {CONTRACT_ADDRESS}
                </code>
                <button
                  onClick={() => copyToClipboard(CONTRACT_ADDRESS, "CA")}
                  className="btn-gradient font-bold px-4 py-3 rounded-xl border border-white/10 transition-transform hover:scale-105 active:scale-95 flex items-center gap-2 flex-shrink-0"
                  data-testid="button-copy-ca"
                >
                  <span aria-hidden="true">⧉</span>
                  Copy
                </button>
              </div>
            </div>
          </div>
        </section>

        <section id="creator-reward" className="py-20">
          <div className="max-w-[1120px] mx-auto px-5">
            <div className="mb-6">
              <h2 className="text-2xl md:text-4xl font-bold tracking-tight font-display" data-testid="text-reward-title">Creator Reward</h2>
            </div>

            <div className="grid md:grid-cols-2 gap-4 mb-4">
              <div className="panel rounded-2xl p-5 hover:border-white/20 transition-colors">
                <div className="flex items-center justify-between gap-3 mb-4">
                  <span className="font-extrabold tracking-tight text-base font-display">Claimed Rewards</span>
                  <a href={PUMP_PROFILE_URL} target="_blank" rel="noreferrer" className="text-white/55 text-sm hover:text-white/90 transition-colors underline decoration-white/20 hover:decoration-white/50 underline-offset-2" data-testid="link-claimed-pump">
                    View on Pump.fun
                  </a>
                </div>
                <div className="panel-dark rounded-xl px-4 py-3">
                  <code className="text-white/90 block font-mono" data-testid="text-claimed-reward">
                    {rewards ? `${rewards.claimedRewards} SOL ($${rewards.claimedRewardsUsd})` : "Loading..."}
                  </code>
                </div>
                <p className="text-white/55 text-sm mt-3">
                  Total creator rewards that have been claimed to the wallet.
                </p>
              </div>

              <div className="panel rounded-2xl p-5 hover:border-white/20 transition-colors">
                <div className="flex items-center justify-between gap-3 mb-4">
                  <span className="font-extrabold tracking-tight text-base font-display">Unclaimed Rewards</span>
                  <a href={PUMP_PROFILE_URL} target="_blank" rel="noreferrer" className="text-white/55 text-sm hover:text-white/90 transition-colors underline decoration-white/20 hover:decoration-white/50 underline-offset-2" data-testid="link-unclaimed-pump">
                    View on Pump.fun
                  </a>
                </div>
                <div className="flex gap-3 items-center">
                  <div className="panel-dark rounded-xl px-4 py-3 flex-1">
                    <code className="text-white/90 block font-mono" data-testid="text-unclaimed-reward">
                      {rewards ? `${rewards.unclaimedRewards} SOL ($${rewards.unclaimedRewardsUsd})` : "Loading..."}
                    </code>
                  </div>
                  <button
                    onClick={() => refetch()}
                    className="h-10 w-10 rounded-lg border border-white/15 bg-white/8 text-white/70 hover:bg-white/12 hover:text-white transition-all flex items-center justify-center flex-shrink-0 active:scale-95"
                    disabled={isFetching}
                    aria-label="Refresh rewards"
                    data-testid="button-refresh-rewards"
                  >
                    {isFetching ? <Loader2 className="w-5 h-5 animate-spin" /> : <span className="text-xl">↻</span>}
                  </button>
                </div>
                <p className="text-white/55 text-sm mt-3">
                  Creator rewards that are available but not yet claimed.
                </p>
              </div>
            </div>

            <div className="panel rounded-2xl p-5 hover:border-white/20 transition-colors relative overflow-hidden snake-neon-box">
              <div className="neon-border-1"></div>
              <div className="neon-border-2"></div>
              <div className="mb-4">
                <span className="font-extrabold tracking-tight text-base font-display">Fees For Buyback</span>
              </div>
              <div className="panel-dark rounded-xl px-4 py-3">
                <code className="text-white/90 block font-mono" data-testid="text-buyback-fees">
                  {rewards ? `${rewards.feesForBuyback} SOL ($${rewards.feesForBuybackUsd})` : "0 SOL ($0.00)"}
                </code>
              </div>
            </div>
          </div>
        </section>

        <section id="tokenomics" className="py-20">
          <div className="max-w-[1120px] mx-auto px-5">
            <div className="mb-6">
              <h2 className="text-2xl md:text-4xl font-bold tracking-tight font-display" data-testid="text-tokenomics-title">Tokenomics</h2>
            </div>

            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
              <StatCard label="Ticker" value="Buy" />
              <StatCard label="Chain" value="Solana" />
              <StatCard label="Name Token" value="BuyBack" />
              <StatCard label="Total Supply" value="2,000,000,000" />
              <StatCard label="Market Cap" value={marketCapData?.marketCapFormatted || "Loading..."} data-testid="text-marketcap" />
              <StatCard label="Mint / Freeze" value="None" />
            </div>
          </div>
        </section>

        <section id="community" className="py-20">
          <div className="max-w-[1120px] mx-auto px-5">
            <div className="mb-6">
              <h2 className="text-2xl md:text-4xl font-bold tracking-tight font-display">Join Community</h2>
              <p className="text-white/70 mt-2">Follow updates, hop into chat, and help shape the next moves.</p>
            </div>

            <div className="grid md:grid-cols-3 gap-4">
              <a href={TWITTER_URL} target="_blank" rel="noreferrer" className="panel rounded-2xl p-5 flex items-center gap-4 hover:bg-white/10 hover:border-white/25 transition-all group" data-testid="link-twitter">
                <div className="w-11 h-11 rounded-xl bg-black/30 border border-white/10 flex items-center justify-center font-extrabold tracking-tight flex-shrink-0 group-hover:scale-110 transition-transform">
                  X
                </div>
                <div>
                  <div className="font-extrabold font-display">X (Twitter)</div>
                  <div className="text-white/55 text-sm mt-1">Community</div>
                </div>
              </a>
              <a href={TELEGRAM_URL} target="_blank" rel="noreferrer" className="panel rounded-2xl p-5 flex items-center gap-4 hover:bg-white/10 hover:border-white/25 transition-all group" data-testid="link-telegram">
                <div className="w-11 h-11 rounded-xl bg-black/30 border border-white/10 flex items-center justify-center font-extrabold tracking-tight flex-shrink-0 group-hover:scale-110 transition-transform">
                  TG
                </div>
                <div>
                  <div className="font-extrabold font-display">Telegram</div>
                  <div className="text-white/55 text-sm mt-1">t.me/buybacksol</div>
                </div>
              </a>
              <a href="https://discord.gg/buyback" target="_blank" rel="noreferrer" className="panel rounded-2xl p-5 flex items-center gap-4 hover:bg-white/10 hover:border-white/25 transition-all group" data-testid="link-discord">
                <div className="w-11 h-11 rounded-xl bg-black/30 border border-white/10 flex items-center justify-center font-extrabold tracking-tight flex-shrink-0 group-hover:scale-110 transition-transform">
                  DS
                </div>
                <div>
                  <div className="font-extrabold font-display">Discord</div>
                  <div className="text-white/55 text-sm mt-1">discord.gg/</div>
                </div>
              </a>
            </div>

            <div className="panel rounded-2xl p-5 mt-4 flex flex-wrap items-center justify-between gap-4 hover:border-white/20 transition-colors">
              <div>
                <h3 className="font-bold text-lg tracking-tight font-display">Ready to ape?</h3>
                <p className="text-white/70 mt-2">Grab the CA, double-check it, and join the community.</p>
              </div>
              <button onClick={() => scrollToSection("copy-ca")} className="btn-gradient font-bold px-4 py-3 rounded-xl border border-white/10 transition-transform hover:scale-105 active:scale-95" data-testid="button-cta-ca">
                Copy CA
              </button>
            </div>
          </div>
        </section>
      </main>

      <footer className="py-6 border-t border-white/8 bg-black/14">
        <div className="max-w-[1120px] mx-auto px-5 flex flex-wrap items-center justify-between gap-4">
          <div className="flex items-center gap-2.5">
            <img src={logoUrl} alt="Buyback logo" width="28" height="28" className="rounded-lg transition-transform duration-300 hover:scale-110" />
            <span className="font-extrabold font-display">Buyback</span>
          </div>
          <div className="flex items-center gap-4 text-white/70 text-sm">
            <span>© {new Date().getFullYear()} Buyback. All rights reserved.</span>
            <span className="border-l border-white/12 pl-4">
              <a href="mailto:dev@buybackonsol.xyz" className="hover:text-white transition-colors">dev@buybackonsol.xyz</a>
            </span>
          </div>
        </div>
      </footer>
    </div>
  );
}

function NavLink({ children, onClick }: { children: React.ReactNode; onClick: () => void }) {
  return (
    <button
      onClick={onClick}
      className="text-white/70 text-sm px-2.5 py-2.5 rounded-xl hover:text-white hover:bg-white/6 transition-colors text-left font-medium"
    >
      {children}
    </button>
  );
}

function StatCard({ label, value, ...props }: { label: string; value: string; [key: string]: any }) {
  return (
    <div className="panel rounded-2xl p-5 hover:border-white/20 transition-colors" {...props}>
      <div className="text-white/55 text-sm">{label}</div>
      <div className="font-extrabold text-xl tracking-tight mt-2 font-display">{value}</div>
    </div>
  );
}
