import { useQuery } from "@tanstack/react-query";

// Types matching shared/schema.ts
export interface MarketCap {
  tokenAddress: string;
  marketCapUsd: string;
  marketCapFormatted: string;
  tokenPriceUsd: string;
  fdv: string;
}

export interface CreatorRewards {
  walletAddress: string;
  claimedRewards: string;
  claimedRewardsUsd: string;
  unclaimedRewards: string;
  unclaimedRewardsUsd: string;
  feesForBuyback: string;
  feesForBuybackUsd: string;
  solPrice: number;
}

// Hook for Market Cap - uses backend API
export function useMarketCap(tokenAddress: string) {
  return useQuery<MarketCap>({
    queryKey: ["marketCap", tokenAddress],
    queryFn: async () => {
      if (!tokenAddress) throw new Error("No token address");
      
      const response = await fetch(`/api/market-cap/${tokenAddress}`);
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to fetch market cap");
      }
      return response.json();
    },
    refetchInterval: 15000,
  });
}

// Hook for Rewards - uses backend API
export function useRewards(walletAddress: string) {
  return useQuery<CreatorRewards>({
    queryKey: ["rewards", walletAddress],
    queryFn: async () => {
      if (!walletAddress) throw new Error("No wallet address");
      
      const response = await fetch(`/api/rewards/${walletAddress}`);
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.message || "Failed to fetch rewards");
      }
      return response.json();
    },
    refetchInterval: 15000,
  });
}
