import type { Express } from "express";
import { createServer, type Server } from "http";
import { Connection, PublicKey, LAMPORTS_PER_SOL } from "@solana/web3.js";
import { OnlinePumpSdk } from "@pump-fun/pump-sdk";

const SOLANA_RPC = "https://api.mainnet-beta.solana.com";
const COINGECKO_API = "https://api.coingecko.com/api/v3/simple/price?ids=solana&vs_currencies=usd";

let cachedSolPrice = 0;
let lastPriceFetch = 0;
const PRICE_CACHE_MS = 30000;

async function fetchSolPrice(): Promise<number> {
  const now = Date.now();
  if (cachedSolPrice > 0 && now - lastPriceFetch < PRICE_CACHE_MS) {
    return cachedSolPrice;
  }

  try {
    const response = await fetch(COINGECKO_API);
    const data = await response.json();
    cachedSolPrice = data?.solana?.usd || 0;
    lastPriceFetch = now;
    return cachedSolPrice;
  } catch (error) {
    console.error("Error fetching SOL price:", error);
    return cachedSolPrice || 0;
  }
}

function lamportsToSol(lamports: bigint): number {
  const solBigInt = lamports / BigInt(LAMPORTS_PER_SOL);
  const remainder = lamports % BigInt(LAMPORTS_PER_SOL);
  const decimalPart = Number(remainder) / LAMPORTS_PER_SOL;
  return Number(solBigInt) + decimalPart;
}

function formatSol(value: number): string {
  return value.toFixed(3);
}

function formatUsd(value: number): string {
  return value.toFixed(2);
}

function formatMarketCap(value: number): string {
  if (value >= 1_000_000) {
    return `$${(value / 1_000_000).toFixed(2)}M`;
  } else if (value >= 1_000) {
    return `$${(value / 1_000).toFixed(2)}K`;
  }
  return `$${value.toFixed(2)}`;
}

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  
  app.get("/api/marketcap/:tokenAddress", async (req, res) => {
    try {
      const { tokenAddress } = req.params;
      
      if (!tokenAddress || tokenAddress.length < 32 || tokenAddress.length > 44) {
        return res.status(400).json({ 
          error: "Invalid token address",
          message: "Please provide a valid Solana token address" 
        });
      }

      // Use DexScreener API to get market cap
      const dexScreenerUrl = `https://api.dexscreener.com/latest/dex/tokens/${tokenAddress}`;
      const dexResponse = await fetch(dexScreenerUrl);
      const dexData = await dexResponse.json();

      let marketCapUsd = 0;
      let tokenPriceUsd = 0;
      let fdv = 0;

      if (dexData.pairs && dexData.pairs.length > 0) {
        // Get the first pair (usually the most liquid)
        const pair = dexData.pairs[0];
        marketCapUsd = pair.marketCap || pair.fdv || 0;
        fdv = pair.fdv || 0;
        tokenPriceUsd = parseFloat(pair.priceUsd) || 0;
      }

      const response = {
        tokenAddress,
        marketCapUsd: marketCapUsd.toFixed(2),
        marketCapFormatted: formatMarketCap(marketCapUsd),
        tokenPriceUsd: tokenPriceUsd.toFixed(8),
        fdv: fdv.toFixed(2),
      };

      res.json(response);
    } catch (error) {
      console.error("Error fetching market cap:", error);
      res.status(500).json({ 
        error: "Failed to fetch market cap",
        message: error instanceof Error ? error.message : "Unknown error occurred"
      });
    }
  });

  app.get("/api/rewards/:walletAddress", async (req, res) => {
    try {
      const { walletAddress } = req.params;
      
      if (!walletAddress || walletAddress.length < 32 || walletAddress.length > 44) {
        return res.status(400).json({ 
          error: "Invalid wallet address",
          message: "Please provide a valid Solana wallet address" 
        });
      }

      let publicKey: PublicKey;
      try {
        publicKey = new PublicKey(walletAddress);
      } catch {
        return res.status(400).json({ 
          error: "Invalid wallet address format",
          message: "The provided address is not a valid Solana public key" 
        });
      }

      const connection = new Connection(SOLANA_RPC, "confirmed");
      const sdk = new OnlinePumpSdk(connection);

      const [solPrice, vaultBalance, volumeStats] = await Promise.all([
        fetchSolPrice(),
        sdk.getCreatorVaultBalanceBothPrograms(publicKey).catch(() => null),
        sdk.fetchUserVolumeAccumulatorTotalStats(publicKey).catch(() => null),
      ]);

      const unclaimedLamports = vaultBalance ? BigInt(vaultBalance.toString()) : BigInt(0);
      const unclaimedSol = lamportsToSol(unclaimedLamports);

      let claimedSol = 0;
      if (volumeStats && volumeStats.totalClaimedTokens) {
        const claimedLamports = BigInt(volumeStats.totalClaimedTokens.toString());
        claimedSol = lamportsToSol(claimedLamports);
      }

      const feesForBuybackSol = 0;

      const response = {
        walletAddress,
        claimedRewards: formatSol(claimedSol),
        claimedRewardsUsd: formatUsd(claimedSol * solPrice),
        unclaimedRewards: formatSol(unclaimedSol),
        unclaimedRewardsUsd: formatUsd(unclaimedSol * solPrice),
        feesForBuyback: formatSol(feesForBuybackSol),
        feesForBuybackUsd: formatUsd(feesForBuybackSol * solPrice),
        solPrice,
      };

      res.json(response);
    } catch (error) {
      console.error("Error fetching creator rewards:", error);
      res.status(500).json({ 
        error: "Failed to fetch rewards",
        message: error instanceof Error ? error.message : "Unknown error occurred"
      });
    }
  });

  return httpServer;
}
